#!/usr/bin/env bash
# shellcheck disable=SC2006
# fail the whole script if a command fails
set -Eeuo pipefail 

cleanup() {
  trap '' EXIT INT TERM ERR
  # script cleanup here
}

trap 'cleanup' EXIT INT TERM ERR

############## Start Utility functions ##############
logToFile() {
    MAX_LOG_LINES=200000
    LINES_TO_REMOVE=10000
    # create the log file directory if it does not exist
    if [[ ! -d "${LOGFILE%/*}" ]]
	then
		mkdir -p "${LOGFILE%/*}"
	fi
    
    # Check if the log file exists
    if [[ -f "$LOGFILE" ]]; then
        # Count the number of lines in the log file
        num_lines=$(wc -l < "$LOGFILE")

        # Check if the number of lines exceeds the limit
        if (( num_lines >= MAX_LOG_LINES )); then
            # Calculate the number of lines to remove
            lines_to_remove=$(( num_lines - MAX_LOG_LINES + LINES_TO_REMOVE ))
            # Remove the older lines from the log file
            sed -i "1,${lines_to_remove}d" "$LOGFILE"
            
            logInfo "Truncated log file $LOGFILE."
        fi
    fi

	echo -e "$1" >> "${LOGFILE}"
}

logInfo() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1  ]]
    then
        calling_funtion_string="${calling_function_name}(): "
    else
        calling_funtion_string=""
    fi

    # check if /dev/tty is available by makeing sure STDOUT is attached to TTY
    if [ -t 1 ] || [ -t 2 ]
	then
		echo -e "[`date +"%Y-%m-%dT%H:%M:%S%z"`] ${BLUE}[Info] ${NOFORMAT}${GREEN}${calling_funtion_string}$1${NOFORMAT}" > /dev/tty
	fi
    
    logToFile "[`date +"%Y-%m-%dT%H:%M:%S%z"`] [Info] ${calling_funtion_string} $1"
}

logWarning() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1  ]]
    then
        calling_funtion_string="${calling_function_name}(): "
    else
        calling_funtion_string=""
    fi

    # check if /dev/tty is available by makeing sure STDOUT is attached to TTY
    if [ -t 1 ] || [ -t 2 ]
	then
		echo -e "[`date +"%Y-%m-%dT%H:%M:%S%z"`] ${ORANGE}[Warning] ${NOFORMAT}${GREEN}${calling_funtion_string}$1${NOFORMAT}" > /dev/tty
	fi
    
    logToFile "[`date +"%Y-%m-%dT%H:%M:%S%z"`] [Warning] ${calling_funtion_string} $1"
}

logError() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1  ]]
    then
        calling_funtion_string="${calling_function_name}(): "
    else
        calling_funtion_string=""
    fi

    # check if /dev/tty is available by makeing sure STDOUT is attached to TTY
    if [ -t 1 ] || [ -t 2 ]
	then
		echo -e "[`date +"%Y-%m-%dT%H:%M:%S%z"`] ${RED}[Error] ${NOFORMAT}${GREEN}${calling_funtion_string}$1${NOFORMAT}" > /dev/tty
	fi
    
    logToFile "[`date +"%Y-%m-%dT%H:%M:%S%z"`] [Error] ${calling_funtion_string} $1"
}

logDebug() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1  ]]
    then
        calling_funtion_string="${calling_function_name}(): "
    else
        calling_funtion_string=""
    fi
    

    if [[ "${debug_flag}" -eq 1  ]]
    then
        # check if /dev/tty is available by makeing sure STDOUT is attached to TTY
        if [ -t 1 ] || [ -t 2 ]
	    then
		    echo -e "[`date +"%Y-%m-%dT%H:%M:%S%z"`] ${PURPLE}[Debug] ${NOFORMAT}${GREEN}${calling_funtion_string}$1${NOFORMAT}" > /dev/tty
	    fi
        logToFile "[`date +"%Y-%m-%dT%H:%M:%S%z"`] [Debug] ${calling_funtion_string}$1"
    fi
}

die() {
  local msg=$1
  local code=${2-1} # default exit status 1
  logError "$msg"
  exit "$code"
}

setup_colors() {
  if [[ -t 2 ]] && [[ -z "${NO_COLOR-}" ]] && [[ "${TERM-}" != "dumb" ]]; then
    NOFORMAT='\033[0m' RED='\033[0;31m' GREEN='\033[0;32m' ORANGE='\033[0;33m' BLUE='\033[0;34m' PURPLE='\033[0;35m' CYAN='\033[0;36m' YELLOW='\033[1;33m'
  else
    NOFORMAT='' RED='' GREEN='' ORANGE='' BLUE='' PURPLE='' CYAN='' YELLOW=''
  fi
}

usage() {
  cat <<EOF
IDSTower Suricata logs deletion script, this script deletes old & unused Suricata logs to prevent the disk from getting full.
The script is normally run by Cron periodically, but can be run manually run for debugging purposes.

Usage: $(basename "${BASH_SOURCE[0]}") [-u] [-k] [-d] [-h] [--no-color]
Available options:

-p, --path                Suricata logs directory path (default: /var/log/suricata)
-u, --disk-utilization-threshold   Disk utilization threshold in percentage (default: 50), this option takes precedence over --keep-days
-k, --keep-days   Number of days to keep logs (default: 90)
-x, --dry-run     Do not delete logs, just print the list of logs that will be deleted
-l, --log-file  Log file path (default: /var/log/idstower/logs_deletion.log)
-d, --debug     Print script debug info
-h, --help      Print this help and exit
--no-color      Do not use colors in output

EOF
  exit
}

parse_params() {
  # default values of variables
  dry_run_flag=0
  debug_flag=0

  while :; do
    case "${1-}" in
    -h | --help) usage ;;
    -x | --dry-run) dry_run_flag=1 ;;
    -d | --debug) debug_flag=1 ;;
    --no-color) NO_COLOR=1 ;;
    -p | --path)
	  SURICATA_LOGS_DIR_PATH="${2-}"
	  shift
	  ;;
    -u | --disk-utilization-threshold)
	  DISK_UTILIZATION_THRESHOLD="${2-}"
      # check if DISK_UTILIZATION_THRESHOLD is a number
      if ! [[ "${DISK_UTILIZATION_THRESHOLD}" =~ ^[0-9]+$ ]]
	  then
		echo -e "disk utilization threshold must be a number" && exit 1
	  fi
      # check if DISK_UTILIZATION_THRESHOLD is between 0 and 100
      if [[ "${DISK_UTILIZATION_THRESHOLD}" -lt 0 ]] || [[ "${DISK_UTILIZATION_THRESHOLD}" -gt 100 ]]
      then
        echo -e "disk utilization threshold must be between 0 and 100" && exit 1
	  fi
	  shift
	  ;;
    -k | --keep-days)
      KEEP_DAYS="${2-}"
      # check if KEEP_DAYS is a positive number
      if ! [[ "${KEEP_DAYS}" =~ ^[0-9]+$ ]]
      then
        echo -e "keep days must be a positive number" && exit 1
	  fi
      shift
      ;;
    -l | --log-file)
	  LOGFILE="${2-}"
	  shift
	  ;;
    -?*) echo "ERROR: Unknown option: $1" & usage ;;
    *) break ;;
    esac
    shift
  done

  # set default values
  [[ -z "${SURICATA_LOGS_DIR_PATH-}" ]] && SURICATA_LOGS_DIR_PATH="/var/log/suricata"
  [[ -z "${DISK_UTILIZATION_THRESHOLD-}" ]] && DISK_UTILIZATION_THRESHOLD="50"
  [[ -z "${KEEP_DAYS-}" ]] && KEEP_DAYS="90"
  [[ -z "${LOGFILE-}" ]] && LOGFILE="/var/log/idstower/logs_deletion.log"

  return 0
}

getDiskUtilization() {
    # get passed disk path
    local disk_path="${1}"
    logDebug "Getting disk utilization for ${disk_path}"
    # get disk utilization
    local disk_utilization=""
    disk_utilization=$(df -h "${disk_path}" | tail -1 | awk '{print $5}' | sed 's/%//g')
    logDebug "Disk utilization for ${disk_path} is: ${disk_utilization}%"
    echo "${disk_utilization}"
}

deleteSuricataLogs() {
    # set passed values
    local suricata_logs_dir_path="${1}"
    local disk_utilization_threshold="${2}"
    local keep_days="${3}"
    local dry_run_flag="${4}"

    # check that Suricata logs directory exists
    logDebug "Verifying that Suricata logs directory exists: ${suricata_logs_dir_path}"
    if [[ ! -d "${1}" ]]
	then
		die "Provided Suricata logs directory does not exist: ${suricata_logs_dir_path}, existing..."
	fi

    local disk_utilization=""
    disk_utilization=$(getDiskUtilization "${suricata_logs_dir_path}")
    logInfo "Current Disk utilization for ${suricata_logs_dir_path} is: ${disk_utilization}%"

    # return 0 if disk utilization is below threshold
    logDebug "Checking if disk utilization is below threshold of ${disk_utilization_threshold}%"
    if [[ "${disk_utilization}" -lt "${disk_utilization_threshold}" ]]
	then
		logInfo "Disk utilization is below threshold, exiting..."
		return 0
	fi

    # check if disk utilization is above threshold
    logDebug "Checking if disk utilization is above threshold of ${disk_utilization_threshold}%"
    if [[ "${disk_utilization}" -gt "${disk_utilization_threshold}" ]]
    then
        logInfo "Disk utilization is above threshold, starting deletion of old Suricata logs..."
		# get list of Suricata logs that are older than keep_days
        logDebug "Getting list of Suricata logs that are older than ${keep_days} days..."
        local suricata_logs_list=""
		suricata_logs_list=$(find "${suricata_logs_dir_path}" -type f -mtime +"${keep_days}")
        logDebug "List of Suricata logs that are older than ${keep_days} days: ${suricata_logs_list}"
        # make sure that these logs are not used by any process
        if [[ ! -z "${suricata_logs_list}" ]]
        then
            logDebug "Checking if Suricata logs are in use..."
            suricata_logs_list=$(lsof -F n "${suricata_logs_list}" | grep -v '^p' | sed 's/^n//g')
            logDebug "List of Suricata logs that are older than ${keep_days} days and not in use: ${suricata_logs_list}"
        fi
		
        # check if there are logs to delete
		if [[ -z "${suricata_logs_list}" ]]
		then
			logInfo "Disk utilization is above the threshold of ${disk_utilization_threshold} and we found no Suricata logs older than ${keep_days} days to delete, thus we will proceed to delete oldest logs until disk utilization is below threshold."
            # start deleting suricata logs one by one starting from the oldest until disk utilization is below threshold
            logInfo "Starting deletion of oldest Suricata logs until disk utilization is below threshold..."
            while [[ "${disk_utilization}" -gt "${disk_utilization_threshold}" ]]
            do
				# get oldest Suricata log
                logDebug "Getting oldest Suricata log..."
                local oldest_suricata_log=""
				oldest_suricata_log=$(find "${suricata_logs_dir_path}" -type f -printf '%T+ %p\n' | sort | head -n 1 | awk '{print $2}')
                logDebug "Oldest Suricata log: ${oldest_suricata_log}"
				
                # make sure that this log is not used by any process
				logDebug "Checking if oldest Suricata log is in use..."
                local oldest_suricata_log_in_use=""
                oldest_suricata_log_in_use=$(lsof -F n "${oldest_suricata_log}" | grep -v '^p' | sed 's/^n//g')
                # check if oldest Suricata log is in use
				if [[ -z "${oldest_suricata_log_in_use}" ]]
				then
					# delete oldest Suricata log
					logInfo "Deleting oldest Suricata log: ${oldest_suricata_log}"
					if [[ "${dry_run_flag}" -eq 1 ]]
					then
						logInfo "Dry run flag is set, will not delete logs, just print the list of logs that will be deleted."
						logInfo "List of Suricata logs that will be deleted:"
						logInfo "${oldest_suricata_log}"
                        break
					else
						logInfo "Deleting oldest Suricata log: ${oldest_suricata_log}"
						rm -f "${oldest_suricata_log}"
					fi
					# get disk utilization
					disk_utilization=$(getDiskUtilization "${suricata_logs_dir_path}")
					logInfo "Current Disk utilization for ${suricata_logs_dir_path} is: ${disk_utilization}%"
				else
					logWarning "Oldest Suricata log: ${oldest_suricata_log} is still in use, will not delete it, make sure that your logshipper is not holding file handles of Suricata logs files after shipping them, exiting..."
					break
				fi
			done
		else
			# delete Suricata logs
			logInfo "Deleting Suricata logs..."
			if [[ "${dry_run_flag}" -eq 1 ]]
			then
				logInfo "Dry run flag is set, script will not delete logs, it will just print the list of logs that will be deleted."
				logInfo "List of Suricata logs that will be deleted:"
				logInfo "${suricata_logs_list}"
			else
				logInfo "Deleting Suricata logs..."
				logInfo "${suricata_logs_list}" | xargs rm -f
			fi
		fi
    fi
}


############## END Utility functions ##############

############## Start Main Script ##############
parse_params "$@"
setup_colors

logInfo "==================== Suricata logs deletion script ===================="
logInfo "Starting script..."

logDebug "Passed parameters: "
logDebug "SURICATA_LOGS_DIR_PATH: ${SURICATA_LOGS_DIR_PATH}"
logDebug "DISK_UTILIZATION_THRESHOLD: ${DISK_UTILIZATION_THRESHOLD}"
logDebug "KEEP_DAYS: ${KEEP_DAYS}"
logDebug "dry_run_flag: ${dry_run_flag}"

# check if lsof command is installed
if ! command -v lsof &> /dev/null
then
    die "lsof command is not installed, please install it before running this script, exiting..."
fi

# delete old Suricata logs
deleteSuricataLogs "${SURICATA_LOGS_DIR_PATH}" "${DISK_UTILIZATION_THRESHOLD}" "${KEEP_DAYS}" "${dry_run_flag}"

logInfo "Finished script"
