#!/usr/bin/env bash
# shellcheck disable=SC2006
# fail the whole script if a command fails
set -Eeuo pipefail

cleanup() {
  trap '' EXIT INT TERM ERR
  # script cleanup here
}

trap 'cleanup' EXIT INT TERM ERR

############## Start Utility functions ##############
logToFile() {
    MAX_LOG_LINES=200000
    LINES_TO_REMOVE=10000
    # create the log file directory if it does not exist
    if [[ ! -d "${LOGFILE%/*}" ]]
	then
		mkdir -p "${LOGFILE%/*}"
	fi

    # Check if the log file exists
    if [[ -f "$LOGFILE" ]]; then
        # Count the number of lines in the log file
        num_lines=$(wc -l < "$LOGFILE")

        # Check if the number of lines exceeds the limit
        if (( num_lines >= MAX_LOG_LINES )); then
            # Calculate the number of lines to remove
            lines_to_remove=$(( num_lines - MAX_LOG_LINES + LINES_TO_REMOVE ))
            # Remove the older lines from the log file
            sed -i "1,${lines_to_remove}d" "$LOGFILE"

            logInfo "Truncated log file $LOGFILE."
        fi
    fi

	echo -e "$1" >> "${LOGFILE}"
}

logInfo() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1  ]]
    then
        calling_function_string="${calling_function_name}(): "
    else
        calling_function_string=""
    fi

    # check if /dev/tty is available by makeing sure STDOUT is attached to TTY
    if [ -t 1 ] || [ -t 2 ]
	then
		echo -e "[`date +"%Y-%m-%dT%H:%M:%S%z"`] ${BLUE}[Info] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" > /dev/tty
	fi

    logToFile "[`date +"%Y-%m-%dT%H:%M:%S%z"`] [Info] ${calling_function_string} $1"
}

logWarning() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1  ]]
    then
        calling_function_string="${calling_function_name}(): "
    else
        calling_function_string=""
    fi

    # check if /dev/tty is available by making sure STDOUT is attached to TTY
    if [ -t 1 ] || [ -t 2 ]
	then
		echo -e "[`date +"%Y-%m-%dT%H:%M:%S%z"`] ${ORANGE}[Warning] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" > /dev/tty
	fi

    logToFile "[`date +"%Y-%m-%dT%H:%M:%S%z"`] [Warning] ${calling_function_string} $1"
}

logError() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1  ]]
    then
        calling_function_string="${calling_function_name}(): "
    else
        calling_function_string=""
    fi

    # check if /dev/tty is available by makeing sure STDOUT is attached to TTY
    if [ -t 1 ] || [ -t 2 ]
	then
		echo -e "[`date +"%Y-%m-%dT%H:%M:%S%z"`] ${RED}[Error] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" > /dev/tty
	fi

    logToFile "[`date +"%Y-%m-%dT%H:%M:%S%z"`] [Error] ${calling_function_string} $1"
}

logDebug() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1  ]]
    then
        calling_function_string="${calling_function_name}(): "
    else
        calling_function_string=""
    fi


    if [[ "${debug_flag}" -eq 1  ]]
    then
        # check if /dev/tty is available by makeing sure STDOUT is attached to TTY
        if [ -t 1 ] || [ -t 2 ]
	    then
		    echo -e "[`date +"%Y-%m-%dT%H:%M:%S%z"`] ${PURPLE}[Debug] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" > /dev/tty
	    fi
        logToFile "[`date +"%Y-%m-%dT%H:%M:%S%z"`] [Debug] ${calling_function_string}$1"
    fi
}

die() {
  local msg=$1
  local code=${2-1} # default exit status 1
  logError "$msg"
  exit "$code"
}

setup_colors() {
  if [[ -t 2 ]] && [[ -z "${NO_COLOR-}" ]] && [[ "${TERM-}" != "dumb" ]]; then
    NOFORMAT='\033[0m' RED='\033[0;31m' GREEN='\033[0;32m' ORANGE='\033[0;33m' BLUE='\033[0;34m' PURPLE='\033[0;35m' CYAN='\033[0;36m' YELLOW='\033[1;33m'
  else
    NOFORMAT='' RED='' GREEN='' ORANGE='' BLUE='' PURPLE='' CYAN='' YELLOW=''
  fi
}

usage() {
  cat <<EOF
IDSTower host heartbeat script, this script send heartbeats and health metrics to IDSTower periodically.
The script is normally run by Cron periodically, but can be run manually run for debugging purposes.

Usage: $(basename "${BASH_SOURCE[0]}") [-u] [-k] [-i] [-s] [-l] [-d] [-h] [--no-color]
Available options:

-u, --api-url   IDSTower API URL (required)
-k, --api-key   IDSTower API Authentication Key (required)
-i, --instances Comma-separated list of Suricata instance names (required)
               Example: "1,2,XYZ" for instance-1, instance-2 and instance-XYZ
-s, --services  List of services to monitor along with their systemd service name (required, example: suricata=suricata.service, filebeat=filebeat.service)
-c, --services-configs Custom config file path for each service (optional, default: suricata=/etc/suricata/suricata.yaml, filebeat=/etc/filebeat/filebeat.yml)
-sld, --suricata-log-dir Suricata log directory path (optional, default: /var/log/suricata)
-l, --log-file  Log file path (optional, default: /var/log/idstower/heartbeat.log)
-d, --debug     Print script debug info
-h, --help      Print this help and exit
--no-color      Do not use colors in output

EOF
  exit
}

parse_params() {
  # default values of variables
  debug_flag=0

  while :; do
    case "${1-}" in
    -h | --help) usage ;;
    -d | --debug) debug_flag=1 ;;
    --no-color) NO_COLOR=1 ;;
    # set APIURL with the passed value
    -u | --api-url)
	  APIURL="${2-}"
	  shift
	  ;;
    # set APIKEY with the passed value
    -k | --api-key)
      APIKEY="${2-}"
      shift
      ;;
    # set instances with the passed value
    -i | --instances)
      INSTANCES="${2-}"
      shift
      ;;
    # set ServicesList with the passed value
    -s | --services)
      # verify that the passed value is not empty
      [[ -z "${2-}" ]] && echo -e "ERROR: Missing value for parameter: -s, --services\n" && usage
      # verify that the passed value is in the correct format
      [[ ! "${2-}" =~ ^([a-zA-Z0-9_]+[[:space:]]*=[[:space:]]*[a-zA-Z0-9_.-]+[[:space:]]*,?[[:space:]]*)+$ ]] && echo -e "ERROR: Invalid value for parameter: -s, --services\n" && usage
      # Declare ServicesList as a global associative array
      declare -g -A ServicesList
      # parse the passed value and split it into an associative array
      IFS=',' read -ra Services <<< "${2-}"
      for service in "${Services[@]}"
      do
          IFS='=' read -r key value <<< "$service"
          # Trim leading and trailing spaces from both key and value
          key=$(echo "$key" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
          value=$(echo "$value" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
          ServicesList["$key"]="$value"
      done
      shift
      ;;
    # set ServicesConfigs with the passed value
    -c | --services-configs)
      # verify that the passed value is not empty
      [[ -z "${2-}" ]] && echo -e "ERROR: Missing value for parameter: -c, --services-configs\n" && usage
      # verify that the passed value is in the correct format
      [[ ! "${2-}" =~ ^([a-zA-Z0-9_]+=[a-zA-Z0-9_/.,-]+[[:space:]]*,[[:space:]]*)*[a-zA-Z0-9_]+=[a-zA-Z0-9_/.,-]+$ ]] && echo -e "ERROR: Invalid value for parameter: -c, --services-configs\n" && usage
      # Declare ServicesConfigsList as a global associative array
      declare -g -A ServicesConfigsList
      # parse the passed value and split it into an associative array
      IFS=',' read -ra ServicesConfigs <<< "${2-}"
      for serviceConfig in "${ServicesConfigs[@]}"
      do
          IFS='=' read -r key value <<< "$serviceConfig"
          # Trim leading and trailing spaces
          key=$(echo "$key" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
          value=$(echo "$value" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
          ServicesConfigsList["$key"]="$value"
      done
      shift
      ;;
    # set suricata log directory path with the passed value
    -sld | --suricata-log-dir)
        SURICATA_LOG_DIR="${2-}"
        # check if the passed value is not empty
        [[ -z "${SURICATA_LOG_DIR-}" ]] && echo -e "ERROR: Missing value for parameter: -ld, --suricata-log-dir\n" && usage
        # verify that the directory exists
        [[ ! -d "${SURICATA_LOG_DIR-}" ]] && echo -e "ERROR: Passed Suricata log directory does not exist: ${SURICATA_LOG_DIR} , Please create it and re-run this script.\n" && usage
        shift
        ;;
    # set LOGFILE with the passed value
    -l | --log-file)
	  LOGFILE="${2-}"
	  shift
	  ;;
    -?*) echo "ERROR: Unknown option: $1" & usage ;;
    *) break ;;
    esac
    shift
  done

  # check required parameters
  [[ -z "${APIURL-}" ]] && echo -e "ERROR: Missing required parameter: -u, --api-url\n" && usage
  [[ -z "${APIKEY-}" ]] && echo -e "ERROR: Missing required parameter: -k, --api-key\n" && usage
  [[ -z "${INSTANCES-}" ]] && echo -e "ERROR: Missing required parameter: -i, --instances\n" && usage
  [[ ${#ServicesList[@]} -eq 0 ]] && echo -e "ERROR: Missing required parameter: -s, --services\n" && usage

  # if services configs was not set, set it to default
  [[ -z "${ServicesConfigsList-}" ]] && declare -g -A ServicesConfigsList=(
    ["suricata"]="/etc/suricata/suricata.yaml"
    ["filebeat"]="/etc/filebeat/filebeat.yml"
  )

  # now set suricata_config_dir based on the passed suricata config file path
  suricata_config_dir="$(dirname "${ServicesConfigsList["suricata"]}")/"

  # if suricata log directory was not set, set it to default
  [[ -z "${SURICATA_LOG_DIR-}" ]] && SURICATA_LOG_DIR="/var/log/suricata"

  # if log file was not set, set it to default
  [[ -z "${LOGFILE-}" ]] && LOGFILE="/var/log/idstower/heartbeat.log"

  return 0
}

getPythonVersion() {
    # Check for python3 command first
    if command -v python3 &>/dev/null; then
        python_version="$(python3 --version | head -n1 | grep -oP '\d+\.\d+\.\d+')"
        echo "$python_version"
    # Fall back to python command if python3 is not available
    elif command -v python &>/dev/null; then
        python_version="$(python --version | head -n1 | grep -oP '\d+\.\d+\.\d+')"
        echo "$python_version"
    else
        echo "Python not found" >&2
        return 1
    fi
}

getHostMetricsInJson() {
    # get host metrics
    logDebug "Getting host metrics..."
    logDebug "Getting CPU usage..."
    cpu_usage=`top -b -n2 | grep "Cpu(s)" | awk '{print $2+$4 "%"}' | tail -n1`
    logDebug "CPU usage: ${cpu_usage}"
    logDebug "Getting memory usage..."
    memory_usage=`free | awk '/Mem/{printf("%.2f%%"), $3/$2*100}'`
    logDebug "Memory usage: ${memory_usage}"
    logDebug "Getting disk usage of Suricata log directory: ${SURICATA_LOG_DIR}"
    # if the SURICATA_LOG_DIR dose not exist, set disk_usage to null and print a warning
    if [[ ! -d "${SURICATA_LOG_DIR}" ]]
    then
        logWarning "Suricata log directory does not exist: ${SURICATA_LOG_DIR}"
        disk_usage="null"
    else
        disk_usage=`df -h "${SURICATA_LOG_DIR}" | tail -n1 | awk '{print $5}'`
    fi
    logDebug "Disk usage: ${disk_usage}"

    logDebug "Building host metrics json..."
    local json_request=""

    # build json string
    json_request+='"hostMetrics": {'
    json_request+='"cpu_usage": "'${cpu_usage}'",'
    json_request+='"memory_usage": "'${memory_usage}'",'
    json_request+='"disk_usage": "'${disk_usage}'"'
    json_request+='}'

    # return json string
    logDebug "host metrics: ${json_request}"
    echo "${json_request}"
}

# Function to get network interfaces as JSON array
getNetworkInterfacesInJson() {
    logDebug "Getting network interfaces..."

    local interfaces_list=""
    local first=1
    
    # Get all network interfaces (including loopback) using ip command
    if command -v ip &>/dev/null; then
        logDebug "Using 'ip' command to get network interfaces..."
        # Get interface names and remove virtual interface suffixes (e.g., @if4980)
        while IFS= read -r interface; do
            if [[ -n "$interface" ]]; then
                # Remove anything after @ symbol (virtual interface suffix)
                interface="${interface%%@*}"
                if [[ $first -eq 1 ]]; then
                    interfaces_list="\"$interface\""
                    first=0
                else
                    interfaces_list="$interfaces_list, \"$interface\""
                fi
                logDebug "Found interface: $interface"
            fi
        done < <(ip -o link show | awk -F': ' '{print $2}')
    elif command -v ifconfig &>/dev/null; then
        logDebug "Using 'ifconfig' command to get network interfaces..."
        # Get interface names
        while IFS= read -r interface; do
            if [[ -n "$interface" ]]; then
                if [[ $first -eq 1 ]]; then
                    interfaces_list="\"$interface\""
                    first=0
                else
                    interfaces_list="$interfaces_list, \"$interface\""
                fi
                logDebug "Found interface: $interface"
            fi
        done < <(ifconfig -a | grep -oP '^[a-zA-Z0-9_-]+(?=:)')
    else
        logWarning "'ip' and 'ifconfig' commands not found, cannot retrieve network interfaces."
        interfaces_list=""
    fi
    
    # Return JSON array
    echo "[$interfaces_list]"
}

# returns suricata binary path
getSuricataBinaryPath(){
	# check if suricata binary exists in the default path
	if [[ -f "/usr/bin/suricata" ]]
	then
		suricata_binary_path="/usr/bin/suricata"
    elif [[ -f "/sbin/suricata" ]]
    then
		suricata_binary_path="/sbin/suricata"
	elif [[ -f "/usr/sbin/suricata" ]]
    then
        suricata_binary_path="/usr/sbin/suricata"
    elif [[ -f "/usr/local/bin/suricata" ]]
	then
		suricata_binary_path="/usr/local/bin/suricata"
	elif [[ -f "/usr/local/sbin/suricata" ]]
	then
		suricata_binary_path="/usr/local/sbin/suricata"
    else
		logWarning "suricata binary not found, please install suricata or make sure it is in the default path."
        suricata_binary_path=""
	fi

    echo "${suricata_binary_path}"
}

# returns suricatasc binary path
getSuricataScBinaryPath() {
    # Common locations across distributions
    local possible_paths=(
        "/usr/bin/suricatasc"
        "/usr/local/bin/suricatasc"
        "/usr/sbin/suricatasc"
        "/usr/local/sbin/suricatasc"
    )

    # First try 'which' command
    local which_result
    which_result=$(which suricatasc 2>/dev/null)
    if [[ -x "${which_result}" ]]; then
        echo "${which_result}"
        return 0
    fi

    # If 'which' fails, check common locations
    for path in "${possible_paths[@]}"; do
        if [[ -x "${path}" ]]; then
            echo "${path}"
            return 0
        fi
    done

    # If nothing found, return empty string
    echo ""
    return 0
}

# Function to get the service name for a specific instance
getSuricataServiceName() {
    local instance_name=$1
    echo "suricata-instance-${instance_name}.service"
}

# Helper function to get instance-specific config file
getInstanceConfigFile() {
    local instance_name=$1
    echo "${suricata_config_dir}instance-${instance_name}/suricata.yaml"
}

# returns suricata Socket file path
getSuricataSocketFile() {
    # get the passed suricata config file path
    local suricata_config_file="${1-}"
    local instance_name="${2-}"

    # Define the instance-specific socket name pattern
    local instance_socket_name=""
    if [[ -n "${instance_name}" ]]; then
        instance_socket_name="suricata-instance-${instance_name}.socket"
        logDebug "Using instance-specific socket name: ${instance_socket_name}"
    fi

    suricata_binary_path=$(getSuricataBinaryPath)
    logDebug "suricata command binary path: ${suricata_binary_path}"

    # first we check if there is a custom socket file path in the config file
    # check if the return suricata binary exists
    logDebug "checking if suricata binary exists..."
    if [[ -f "${suricata_binary_path}" ]]
    then
        logDebug "obtaining socket file path from Suricata config at: ${suricata_config_file}..."
        socket_file=$(${suricata_binary_path} -c "${suricata_config_file}" --dump-config | grep unix-command.filename)
        # if exit code is 0, then there is a custom socket file path in the config file
        if [[ $? -eq 0 ]]
        then
            # get the socket file path from the config file
            socket_file=$(echo "${socket_file}" | cut -d "=" -f2 | tr -d ' ')

            # check if socket_file is empty or not
            if [[ -z "${socket_file}" ]]
            then
                logDebug "socket file not found in Suricata config file: ${suricata_config_file}"
            else
                logDebug "socket file found in Suricata config file: ${socket_file}"
            fi

            logDebug "checking if socket file exists..."
            # check if socket file exists
            if [[ -S "${socket_file}" ]]
            then
                logDebug "Suricata socket file found: ${socket_file}"
            elif [[ -S "/var/${socket_file}" ]]
            then
                socket_file="/var/${socket_file}"
                logDebug "socket file found: ${socket_file}"
            elif [[ -S "/var/run/${socket_file}" ]]
            then
                socket_file="/var/run/${socket_file}"
                logDebug "socket file found: ${socket_file}"
            elif [[ -S "/var/run/suricata/${socket_file}" ]]
            then
                socket_file="/var/run/suricata/${socket_file}"
                logDebug "socket file found: ${socket_file}"
            # Check instance-specific socket paths if an instance name is provided
            elif [[ -n "${instance_name}" ]] && [[ -S "/var/${instance_socket_name}" ]]
            then
                socket_file="/var/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/${instance_socket_name}" ]]
            then
                socket_file="/var/run/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/suricata/${instance_socket_name}" ]]
            then
                socket_file="/var/run/suricata/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            else
                logWarning "socket file for instance ${instance_name} was not found: ${socket_file}"
                echo ""
                return 0
            fi
        else
            logDebug "socket file not found in Suricata config file: ${suricata_config_file}"
            logDebug "checking if instance-specific or default socket file exists..."

            # Check for instance-specific socket files first if an instance name is provided
            if [[ -n "${instance_name}" ]] && [[ -S "/var/${instance_socket_name}" ]]
            then
                socket_file="/var/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/${instance_socket_name}" ]]
            then
                socket_file="/var/run/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/suricata/${instance_socket_name}" ]]
            then
                socket_file="/var/run/suricata/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            else
                logWarning "socket file for instance ${instance_name} was not found: ${socket_file}"
                echo ""
                return 0
            fi
        fi
    elif [[ -n "${instance_name}" ]] && [[ -S "/var/${instance_socket_name}" ]]
    then
        socket_file="/var/${instance_socket_name}"
        logDebug "Suricata binary path not found, using instance socket file path: ${socket_file}"
    elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/${instance_socket_name}" ]]
    then
        socket_file="/var/run/${instance_socket_name}"
        logDebug "Suricata binary path not found, using instance socket file path: ${socket_file}"
    elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/suricata/${instance_socket_name}" ]]
    then
        socket_file="/var/run/suricata/${instance_socket_name}"
        logDebug "Suricata binary path not found, using instance socket file path: ${socket_file}"
    else
        logWarning "Suricata binary path not found, please install suricata or make sure it is in the default path."
        echo ""
        return 0
    fi

    # after we have got the socket file path, we need to check if it is active (eg: the service is running)
    # to do this we run a simple command and observe the exit code
    # if the exit code is 0, then the socket file is active
    logDebug "getting suricatasc binary path..."
    suricatasc_binary_path=$(getSuricataScBinaryPath)
    if [[ -z "${suricatasc_binary_path}" ]]; then
       logError "Could not find suricatasc executable"
       echo ""
       return 1
    fi

    logDebug "suricatasc command binary path: ${suricatasc_binary_path}"
    logDebug "checking if Suricata socket file at the following location is active: ${socket_file}"

    "${suricatasc_binary_path}" -c version "${socket_file}" > /dev/null 2>&1
    if [[ $? -eq 0 ]]; then
       logDebug "the following socket file is active: ${socket_file}"
       echo "${socket_file}"
       return 0
    else
       logWarning "socket file for instance ${instance_name} is not active: ${socket_file}, make sure suricata service for instance ${instance_name} is running."
       echo ""
       return 0
    fi
}

# returns suricata version
getSuricataVersion() {
    # get passed suricata config file path
    suricata_config_file="${1-}"
    # get instance name (new parameter)
    instance_name="${2-}"
    logDebug "Suricata config file path: ${suricata_config_file}"
    logDebug "Instance name: ${instance_name}"

    logDebug "Getting Suricata version using suricatasc..."
    # get suricata socket file path, passing the instance name
    socket_file=$(getSuricataSocketFile "${suricata_config_file}" "${instance_name}")
    logDebug "Suricata socket file path: ${socket_file}"

    suricata_binary_path=$(getSuricataBinaryPath)
    logDebug "suricata command binary path: ${suricata_binary_path}"

    suricatasc_binary_path=$(getSuricataScBinaryPath)
    logDebug "suricatasc command binary path: ${suricatasc_binary_path}"

    # check if socket file is not empty & exists
    if [[ -S "${socket_file}" ]]
    then
        logDebug "Suricata socket file found: ${socket_file}"
        logDebug "Getting Suricata version using suricatasc..."
        suricata_version=$("${suricatasc_binary_path}" -c version "${socket_file}" | cut -d "\"" -f 4 | grep -oP '\d+\.\d+\.\d+')
    elif [[ -f "${suricata_binary_path}" ]]
    then
        logWarning "No active Suricata socket file found, Suricata service might be stopped."
        logDebug "obtaining suricata version using suricata command"
        suricata_version=$(${suricata_binary_path} -V | head -n1 | grep -oP '\d+\.\d+\.\d+')
    else
        logWarning "Could not obtain Suricata version."
        suricata_version=""
    fi

    # return suricata version
    logDebug "Suricata version: ${suricata_version}"
    echo "${suricata_version}"
}

# check for new Suricata versions
getAvailableSuricataVersionsInRepositories() {
    local CURRENT_VERSION="$1"
    local VERSIONS=()
    current_hour=$(date +%H)
    current_minute=$(date +%M)

    logDebug "running getAvailableSuricataVersionsInRepositories, current Suricata version: ${CURRENT_VERSION}"
    logDebug "Current time: ${current_hour}:${current_minute}"

    if command -v dnf &> /dev/null; then
        logDebug "Using DNF package manager"
        # Run update if it's between 00:00 and 00:05 only
        if [ "$current_hour" == "00" ] && [ "$current_minute" -ge "00" ] && [ "$current_minute" -le "05" ]; then
            logDebug "It's shortly after midnight. Running apt-get update..."
            dnf check-update >/dev/null 2>&1
        else
            logDebug "Not update time. Skipping apt-get update."
        fi
        # Check default repositories
        mapfile -t VERSIONS < <(dnf list --available suricata 2>/dev/null | grep suricata | awk '{print $2}' | grep -oP '\d+\.\d+\.\d+' | sort -V | uniq)

    elif command -v yum &> /dev/null; then
        logDebug "Using YUM package manager"
        # Run update if it's between 00:00 and 00:05 only
        if [ "$current_hour" == "00" ] && [ "$current_minute" -ge "00" ] && [ "$current_minute" -le "05" ]; then
            logDebug "It's shortly after midnight. Running apt-get update..."
            yum check-update >/dev/null 2>&1
        else
            logDebug "Not update time. Skipping apt-get update."
        fi
        # Check default repositories
        mapfile -t VERSIONS < <(yum list --available suricata 2>/dev/null | grep suricata | awk '{print $2}' | grep -oP '\d+\.\d+\.\d+' | sort -V | uniq)

    elif command -v apt-get &> /dev/null; then
        logDebug "Using APT package manager"
        # Run update if it's between 00:00 and 00:05 only
        if [ "$current_hour" == "00" ] && [ "$current_minute" -ge "00" ] && [ "$current_minute" -le "05" ]; then
            logDebug "It's shortly after midnight. Running apt-get update..."
            apt-get update >/dev/null 2>&1
        else
            logDebug "Not update time. Skipping apt-get update."
        fi
        mapfile -t VERSIONS < <(apt-cache policy suricata 2>/dev/null | grep -oP '\d+\.\d+\.\d+' | sort -V | uniq)
    else
        logWarning "No supported package manager found (dnf, yum, or apt-get)"
    fi

    # Remove duplicates
    mapfile -t VERSIONS < <(printf "%s\n" "${VERSIONS[@]}" | sort -V | uniq)

    logDebug "Available Suricata versions: ${VERSIONS[*]}"

    echo "${VERSIONS[@]}"
}

# check for new Filebeat versions
getAvailableFilebeatVersionsInRepositories() {
    local CURRENT_VERSION="$1"
    local VERSIONS=()
    current_hour=$(date +%H)
    current_minute=$(date +%M)

    logDebug "Running getAvailableFilebeatVersionsInRepositories, current Filebeat version: ${CURRENT_VERSION}"
    logDebug "Current time: ${current_hour}:${current_minute}"

    if command -v dnf &> /dev/null; then
        logDebug "Using DNF package manager"
        # Run update if it's between 00:00 and 00:05 only
        if [ "$current_hour" == "00" ] && [ "$current_minute" -ge "00" ] && [ "$current_minute" -le "05" ]; then
            logDebug "It's shortly after midnight. Running dnf update..."
            dnf check-update >/dev/null 2>&1
        else
            logDebug "Not update time. Skipping dnf update."
        fi
        # Check default repositories
        mapfile -t VERSIONS < <(dnf list --available filebeat 2>/dev/null | grep filebeat | awk '{print $2}' | grep -oP '\d+\.\d+\.\d+' | sort -V | uniq)

    elif command -v yum &> /dev/null; then
        logDebug "Using YUM package manager"
        # Run update if it's between 00:00 and 00:05 only
        if [ "$current_hour" == "00" ] && [ "$current_minute" -ge "00" ] && [ "$current_minute" -le "05" ]; then
            logDebug "It's shortly after midnight. Running yum update..."
            yum check-update >/dev/null 2>&1
        else
            logDebug "Not update time. Skipping yum update."
        fi
        # Check default repositories
        mapfile -t VERSIONS < <(yum list --available filebeat 2>/dev/null | grep filebeat | awk '{print $2}' | grep -oP '\d+\.\d+\.\d+' | sort -V | uniq)

    elif command -v apt-get &> /dev/null; then
        logDebug "Using APT package manager"
        # Run update if it's between 00:00 and 00:05 only
        if [ "$current_hour" == "00" ] && [ "$current_minute" -ge "00" ] && [ "$current_minute" -le "05" ]; then
            logDebug "It's shortly after midnight. Running apt-get update..."
            apt-get update >/dev/null 2>&1
        else
            logDebug "Not update time. Skipping apt-get update."
        fi
        mapfile -t VERSIONS < <(apt-cache policy filebeat 2>/dev/null | grep -oP '\d+\.\d+\.\d+' | sort -V | uniq)
    else
        logWarning "No supported package manager found (dnf, yum, or apt-get)"
    fi

    # Remove duplicates
    mapfile -t VERSIONS < <(printf "%s\n" "${VERSIONS[@]}" | sort -V | uniq)

    logDebug "Available Filebeat versions: ${VERSIONS[*]}"

    echo "${VERSIONS[@]}"
}

# Function to build complete Suricata service status with all instances
buildSuricataServiceStatus() {
    logInfo "Building Suricata service status with all instances..."

    # Get current Suricata version
    local suricata_binary_path
    suricata_binary_path=$(getSuricataBinaryPath)
    local current_service_version=""

    if [[ -f "${suricata_binary_path}" ]]; then
        current_service_version=$(${suricata_binary_path} -V | head -n1 | grep -oP '\d+\.\d+\.\d+')
        logDebug "Current Suricata version: ${current_service_version}"
    else
        logWarning "Suricata binary not found, version information will be empty"
    fi

    # Get available versions
    logDebug "Checking available Suricata versions..."
    local available_service_versions
    available_service_versions=$(getAvailableSuricataVersionsInRepositories "${current_service_version}")

    # Split the versions string into an array and format for JSON
    IFS=' ' read -ra version_array <<< "$available_service_versions"
    local formatted_new_versions=""
    if [ ${#version_array[@]} -eq 0 ]; then
        formatted_new_versions=""
    else
        formatted_new_versions=$(printf '"%s",' "${version_array[@]}" | sed 's/,$//')
    fi

    # Process all instances to build JSON array
    local instances_json="["
    local instance_count=0

    # Parse instances from the comma-separated list
    IFS=',' read -ra INSTANCE_ARRAY <<< "$INSTANCES"

    # Process each instance
    for instance_name in "${INSTANCE_ARRAY[@]}"
    do
        # Trim whitespace from instance name
        instance_name=$(echo "$instance_name" | xargs)

        if [[ -z "$instance_name" ]]; then
            logWarning "Empty instance name found in list, skipping"
            continue
        fi

        logInfo "Processing Suricata instance: ${instance_name}"

        # Get instance-specific configuration file
        local instance_config_file
        instance_config_file=$(getInstanceConfigFile "${instance_name}")

        # If instance-specific config doesn't exist, use default
        if [[ ! -f "${instance_config_file}" ]]; then
            logWarning "Instance-specific config file for ${instance_name} not found, using default config"
            instance_config_file="${ServicesConfigsList["suricata"]}"
        fi

        # Get instance-specific service name
        local service_name
        service_name=$(getSuricataServiceName "${instance_name}")

        # Get service status
        local service_status
        service_status=$(systemctl is-active "${service_name}")
        logDebug "Service status for ${service_name}: ${service_status}"

        # Get uptime information, if service is active, if not set to null
        if [[ "${service_status}" == "active" ]]; then
            local service_uptime_day
            service_uptime_day=$(systemctl status "${service_name}" 2>/dev/null | grep "Active:" | sed 's/.*since //' | cut -d' ' -f1)
            local service_uptime_hours
            service_uptime_hours=$(systemctl status "${service_name}" 2>/dev/null | grep "Active:" | sed 's/.*since //' | cut -d' ' -f2)
            local service_uptime_timezone
            service_uptime_timezone=$(systemctl status "${service_name}" 2>/dev/null | grep "Active:" | sed 's/.*since //' | cut -d' ' -f3 | sed 's/.$//')
            local service_uptime="\"${service_uptime_day} ${service_uptime_hours} ${service_uptime_timezone}\""
            logDebug "Service uptime for ${service_name}: ${service_uptime}"
        else
            service_uptime="null"
            logDebug "Service ${service_name} is not active, setting uptime to null"
        fi

        # Get suricata Socket file path
        local socket_file
        socket_file=$(getSuricataSocketFile "${instance_config_file}" "${instance_name}")
        logDebug "Suricata socket file for ${instance_name}: ${socket_file}"

        # Get suricata metrics using the socket if available
        local suricata_stats_json="null"

        if [[ -S "${socket_file}" ]]; then
            local suricatasc_binary_path
            suricatasc_binary_path=$(getSuricataScBinaryPath)

            # Get rules stats
            local suricata_rules_stats
            suricata_rules_stats=$("${suricatasc_binary_path}" -c ruleset-stats "${socket_file}")
            if [[ "$suricata_rules_stats" == "" ]]; then
                suricata_rules_stats="null"
            fi

            # Get counters (two measurements over 1 second)
            local suricata_counters_t0
            suricata_counters_t0=$("${suricatasc_binary_path}" -c dump-counters "${socket_file}")
            if [[ "$suricata_counters_t0" == "" ]]; then
                suricata_counters_t0="null"
            fi

            sleep 1

            local suricata_counters_t1
            suricata_counters_t1=$("${suricatasc_binary_path}" -c dump-counters "${socket_file}")
            if [[ "$suricata_counters_t1" == "" ]]; then
                suricata_counters_t1="null"
            fi

            # Build metrics JSON
            suricata_stats_json="{"
            suricata_stats_json+='"suricata_rules_stats": '${suricata_rules_stats}','
            suricata_stats_json+='"suricata_counters": {'
            suricata_stats_json+='"t0": '${suricata_counters_t0}','
            suricata_stats_json+='"t1": '${suricata_counters_t1}''
            suricata_stats_json+='}'
            suricata_stats_json+='}'
        else
            logWarning "Cannot obtain metrics for instance ${instance_name}, active socket file not found."
        fi

        # Add separator between instances if not the first
        if [[ $instance_count -gt 0 ]]; then
            instances_json+=","
        fi

        # Build instance status JSON
        instances_json+="{"
        instances_json+='"instanceName": "'${instance_name}'",'
        instances_json+='"status": "'${service_status}'",'
        instances_json+='"uptime": '${service_uptime}','
        instances_json+='"metrics": '${suricata_stats_json}''
        instances_json+="}"

        ((instance_count++))
    done

    # Close the instances array
    instances_json+="]"

    logInfo "Processed ${instance_count} Suricata instances"

    # Build complete service status JSON
    local service_json="{"
    service_json+='"name": "Suricata",'
    service_json+='"version": "'${current_service_version}'",'
    service_json+='"availableVersions": ['${formatted_new_versions}'],'
    service_json+='"instances": '${instances_json}''
    service_json+='}'

    echo "${service_json}"
}

# Updated function for Filebeat service status
buildFilebeatServiceStatus() {
    logInfo "Building Filebeat service status..."

    # Get Filebeat version
    local current_service_version=""
    if ! command -v filebeat &> /dev/null; then
        logWarning "Filebeat command not found, version information will be empty"
    else
        current_service_version=$(filebeat version | cut -d " " -f 3)
        logDebug "Filebeat version: ${current_service_version}"
    fi

    # Get available versions
    logDebug "Checking available Filebeat versions..."
    local available_service_versions
    available_service_versions=$(getAvailableFilebeatVersionsInRepositories "${current_service_version}")

    # Split the versions string into an array and format for JSON
    IFS=' ' read -ra version_array <<< "$available_service_versions"
    local formatted_new_versions=""
    if [ ${#version_array[@]} -eq 0 ]; then
        formatted_new_versions=""
    else
        formatted_new_versions=$(printf '"%s",' "${version_array[@]}" | sed 's/,$//')
    fi

    # Get service status
    local service_name="${ServicesList["filebeat"]}"
    local service_status
    service_status=$(systemctl is-active "${service_name}")
    logDebug "Filebeat service status: ${service_status}"

    local service_uptime_day
    service_uptime_day=$(systemctl status "${service_name}" 2>/dev/null | grep "Active:" | sed 's/.*since //' | cut -d' ' -f1)
    local service_uptime_hours
    service_uptime_hours=$(systemctl status "${service_name}" 2>/dev/null | grep "Active:" | sed 's/.*since //' | cut -d' ' -f2)
    local service_uptime_timezone
    service_uptime_timezone=$(systemctl status "${service_name}" 2>/dev/null | grep "Active:" | sed 's/.*since //' | cut -d' ' -f3 | sed 's/.$//')
    local service_uptime="${service_uptime_day} ${service_uptime_hours} ${service_uptime_timezone}"

    # Build instance JSON (single instance for Filebeat)
    local instance_json="{"
    instance_json+='"instanceName": "default",'
    instance_json+='"status": "'${service_status}'",'
    instance_json+='"uptime": "'${service_uptime}'",'
    instance_json+='"metrics": null'
    instance_json+='}'

    # Build complete service status JSON
    local service_json="{"
    service_json+='"name": "Filebeat",'
    service_json+='"version": "'${current_service_version}'",'
    service_json+='"availableVersions": ['${formatted_new_versions}'],'
    service_json+='"instances": ['${instance_json}']'
    service_json+='}'

    echo "${service_json}"
}

############## END Utility functions ##############

############## Start Main Script ##############
parse_params "$@"
setup_colors

# Ensure common system binary paths are in PATH for cron environment
# Append (not prepend) to avoid PATH hijacking security issues when running as root
logDebug "Ensuring common system binary paths are in PATH..."
logDebug "Original PATH: $PATH"
for path_dir in /sbin /usr/sbin /usr/local/sbin /bin /usr/bin /usr/local/bin; do
    if [[ -d "$path_dir" ]] && [[ ":$PATH:" != *":$path_dir:"* ]]; then
        export PATH="${PATH}:${path_dir}"
    fi
done
logDebug "PATH is set to: $PATH"

logInfo "==================== IDSTower Heartbeat Script ===================="
logInfo "Starting script..."

# check if curl exits
logDebug "Verifying curl command exists..."
if ! command -v curl &> /dev/null
then
	logError "curl command could not be found, please install it and try again."
	exit 1
fi

# count number of services in ServicesList
num_services=${#ServicesList[@]}
logDebug "Number of services: ${num_services}"
# loop thru the associative array and print the key and value
logDebug "Services List:"
for key in "${!ServicesList[@]}"
do
    logDebug "Service: ${key}, Service Name: ${ServicesList[$key]}"
done

# start building the json request
logDebug "Building json request..."
json_request="{"
json_request+="\"apikey\": \"${APIKEY}\","

# add host Architecture to json request
logDebug "Getting host architecture..."
host_architecture=`uname -m`
logDebug "Host Architecture: ${host_architecture}"
json_request+="\"Architecture\": \"${host_architecture}\","

# add Distribution to json request
logDebug "Getting host distribution..."
OS_DISTRIBUTION="$(. /etc/os-release && echo "$ID"| tr '[:upper:]' '[:lower:]')"
logDebug "OS_DISTRIBUTION: $OS_DISTRIBUTION"
json_request+="\"Distribution\": \"${OS_DISTRIBUTION}\","

# add Distribution version to json request
OS_DISTRIBUTION_VER="$(. /etc/os-release && echo "$VERSION_ID")"
logDebug "OS_DISTRIBUTION_VER: $OS_DISTRIBUTION_VER"
json_request+="\"DistributionVersion\": \"${OS_DISTRIBUTION_VER}\","

# add kernel name to json request
kernel_name=`uname -s`
logDebug "Kernel Name: ${kernel_name}"
json_request+="\"Kernel\": \"${kernel_name}\","

# add kernel version to json request
kernel_version=`uname -r`
logDebug "Kernel Version: ${kernel_version}"
json_request+="\"KernelVersion\": \"${kernel_version}\","

# add host python version to json request
logDebug "Getting Python version..."
python_version=`getPythonVersion`
logDebug "Python version: ${python_version}"
json_request+="\"PythonVersion\": \"${python_version}\","

# add host Network Interfaces to json request
logInfo "Getting host network interfaces..."
network_interfaces_json=`getNetworkInterfacesInJson`
logDebug "Network interfaces: ${network_interfaces_json}"
json_request+="\"NetworkInterfaces\": ${network_interfaces_json},"

# add host metrics to json request
logInfo "Getting host metrics..."
host_metrics_json=`getHostMetricsInJson`
json_request+="${host_metrics_json}"

# check if num_services is greater than 0
if [[ ${num_services} -gt 0 ]]
then
    json_request+=","
fi

# build services status json string
json_request+='"servicesStatus": ['

i=0
for service in "${!ServicesList[@]}"
do
    let i=i+1

    if [[ "$service" == "suricata" ]]
    then
        logInfo "Getting Suricata service status with all instances..."
        suricata_service_status=$(buildSuricataServiceStatus)
        json_request+="${suricata_service_status}"

        if [[ $i -ne $num_services ]]; then
            json_request+=","
        fi
    elif [[ "$service" == "filebeat" ]]
    then
        logInfo "Getting Filebeat service status..."
        filebeat_service_status=$(buildFilebeatServiceStatus)
        json_request+="${filebeat_service_status}"

        if [[ $i -ne $num_services ]]; then
            json_request+=","
        fi
    else
        die "Unknown target software: ${service}"
    fi
done

json_request+=']'

json_request+="}"

# save the json request to a file
logDebug "Saving json request to a temporary file: /tmp/idstower_heartbeat.tmp"
echo "${json_request}" > /tmp/idstower_heartbeat.tmp

# send the request
logInfo "Sending heartbeat to IDSTower..."
if [[ "${debug_flag}" -eq 1 ]]
then
	apiResponse=`curl -v -L -s -k --max-time 10 -X POST -H "Content-Type: application/json" --data "@/tmp/idstower_heartbeat.tmp" "${APIURL}"hosts/heartbeat || true`
else
	apiResponse=`curl -L -s -k --max-time 10 -X POST -H "Content-Type: application/json" --data "@/tmp/idstower_heartbeat.tmp" "${APIURL}"hosts/heartbeat || true`
fi

# check if api response is not empty
if [[ -z "${apiResponse}" ]]
then
	logError "Failed to send heartbeat to IDSTower, Make sure IDSTower service is running, to debug this issue, run the script with -d flag."
else
    logInfo "Heartbeat sent successfully to IDSTower, API Response: ${apiResponse}"
fi

# delete the json request file
logDebug "Deleting temporary file: /tmp/idstower_heartbeat.tmp"
rm -rf /tmp/idstower_heartbeat.tmp

logInfo "IDSTower Heartbeat finished."
