#!/usr/bin/env bash
# shellcheck disable=SC2006
# fail the whole script if a command fails
set -Eeuo pipefail

cleanup() {
    trap '' EXIT INT TERM ERR
    # script cleanup here
}

trap 'cleanup' EXIT INT TERM ERR

############## Start Utility functions ##############
logToFile() {
    MAX_LOG_LINES=200000
    LINES_TO_REMOVE=10000
    # create the log file directory if it does not exist
    if [[ ! -d "${LOGFILE%/*}" ]]; then
        mkdir -p "${LOGFILE%/*}"
    fi

    # Check if the log file exists
    if [[ -f "$LOGFILE" ]]; then
        # Count the number of lines in the log file
        num_lines=$(wc -l <"$LOGFILE")

        # Check if the number of lines exceeds the limit
        if ((num_lines >= MAX_LOG_LINES)); then
            # Calculate the number of lines to remove
            lines_to_remove=$((num_lines - MAX_LOG_LINES + LINES_TO_REMOVE))
            # Remove the older lines from the log file
            sed -i "1,${lines_to_remove}d" "$LOGFILE"

            logInfo "Truncated log file $LOGFILE."
        fi
    fi

    echo -e "$1" >>"${LOGFILE}"
}

logInfo() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1 ]]; then
        calling_function_string="${calling_function_name}(): "
    else
        calling_function_string=""
    fi

    # check if /dev/tty is available by making sure STDOUT is attached to TTY
    if [ -t 1 ] || [ -t 2 ]; then
        echo -e "[$(date +"%Y-%m-%dT%H:%M:%S%z")] ${BLUE}[Info] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" >/dev/tty
    fi

    logToFile "[$(date +"%Y-%m-%dT%H:%M:%S%z")] [Info] ${calling_function_string} $1"
}

logWarning() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1 ]]; then
        calling_function_string="${calling_function_name}(): "
    else
        calling_function_string=""
    fi

    # check if /dev/tty is available by making sure STDOUT is attached to TTY
    if [ -t 1 ] || [ -t 2 ]; then
        echo -e "[$(date +"%Y-%m-%dT%H:%M:%S%z")] ${ORANGE}[Warning] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" >/dev/tty
    fi

    logToFile "[$(date +"%Y-%m-%dT%H:%M:%S%z")] [Warning] ${calling_function_string} $1"
}

logError() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1 ]]; then
        calling_function_string="${calling_function_name}(): "
    else
        calling_function_string=""
    fi

    # check if /dev/tty is available by making sure STDOUT is attached to TTY
    if [ -t 1 ] || [ -t 2 ]; then
        echo -e "[$(date +"%Y-%m-%dT%H:%M:%S%z")] ${RED}[Error] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" >/dev/tty
    fi

    logToFile "[$(date +"%Y-%m-%dT%H:%M:%S%z")] [Error] ${calling_function_string} $1"
}

logDebug() {
    local calling_function_name=${FUNCNAME[1]}
    if [[ "${debug_flag}" -eq 1 ]]; then
        calling_function_string="${calling_function_name}(): "
    else
        calling_function_string=""
    fi

    if [[ "${debug_flag}" -eq 1 ]]; then
        # check if /dev/tty is available by making sure STDOUT is attached to TTY
        if [ -t 1 ] || [ -t 2 ]; then
            echo -e "[$(date +"%Y-%m-%dT%H:%M:%S%z")] ${PURPLE}[Debug] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" >/dev/tty
        fi
        logToFile "[$(date +"%Y-%m-%dT%H:%M:%S%z")] [Debug] ${calling_function_string}$1"
    fi
}

die() {
    local msg=$1
    local code=${2-1} # default exit status 1
    logError "$msg"
    exit "$code"
}

setup_colors() {
    if [[ -t 2 ]] && [[ -z "${NO_COLOR-}" ]] && [[ "${TERM-}" != "dumb" ]]; then
        NOFORMAT='\033[0m' RED='\033[0;31m' GREEN='\033[0;32m' ORANGE='\033[0;33m' BLUE='\033[0;34m' PURPLE='\033[0;35m' CYAN='\033[0;36m' YELLOW='\033[1;33m'
    else
        NOFORMAT='' RED='' GREEN='' ORANGE='' BLUE='' PURPLE='' CYAN='' YELLOW=''
    fi
}

usage() {
    cat <<EOF
IDSTower Suricata rules update script, this script download latest Suricata rules from IDSTower periodically and applies them to Suricata.
The script is normally run by Cron periodically, but can be run manually run for debugging purposes.

Usage: $(basename "${BASH_SOURCE[0]}") [-u] [-k] [-i] [-f] [-s] [-l] [-d] [-h] [--no-color]
Available options:

-u, --api-url   IDSTower API URL (default: https://IDSTower_Host/api/v1/)
-k, --api-key   IDSTower API Authentication Key
-i, --instances Comma-separated list of instance names (required)
               Example: "1,2,XYZ" for instance-1, instance-2 and instance-XYZ
-cd, --suricata-config-dir Suricata configuration directory (default: /etc/suricata/)
-f, --force     Force Update even if we have the latest Rules version locally
-s, --skip-delay Skips the initial random delay before running the update
-l, --log-file  Log file path (default: /var/log/idstower/rules_updates.log)
-d, --debug     Print script debug info
-h, --help      Print this help and exit
--no-color      Do not use colors in output

EOF
    exit
}

parse_params() {
    # default values of variables
    debug_flag=0
    force_flag=0
    skip_delay_flag=0

    while :; do
        case "${1-}" in
        -h | --help) usage ;;
        -d | --debug) debug_flag=1 ;;
        -f | --force) force_flag=1 ;;
        -s | --skip-delay) skip_delay_flag=1 ;;
        --no-color) NO_COLOR=1 ;;
        # set APIURL with the passed value
        -u | --api-url)
            APIURL="${2-}"
            shift
            ;;
        # set APIKEY with the passed value
        -k | --api-key)
            APIKEY="${2-}"
            shift
            ;;
        # set instances with the passed value
        -i | --instances)
            INSTANCES="${2-}"
            shift
            ;;
        # set suricata config dir with the passed value
        -cd | --suricata-config-dir)
            suricata_config_dir="${2-}"
            # verify that the passed suricata config dir exists
            if [[ ! -d "${suricata_config_dir}" ]]; then
                echo -e "Provided Suricata config directory does not exist: ${suricata_config_dir}, exiting..." && exit 1
            fi
            shift
            ;;

        # set LOGFILE with the passed value
        -l | --log-file)
            LOGFILE="${2-}"
            shift
            ;;
        -?*)
            echo "ERROR: Unknown option: $1" &
            usage
            ;;
        *) break ;;
        esac
        shift
    done

    # check required params api-url and api-Key and instances
    [[ -z "${APIURL-}" ]] && echo -e "ERROR: Missing required parameter: -u, --api-url\n" && usage
    [[ -z "${APIKEY-}" ]] && echo -e "ERROR: Missing required parameter: -k, --api-key\n" && usage
    [[ -z "${INSTANCES-}" ]] && echo -e "ERROR: Missing required parameter: -i, --instances\n" && usage

    # if suricata config dir was not set, set it to default
    [[ -z "${suricata_config_dir-}" ]] && suricata_config_dir="/etc/suricata/"

    # if log file was not set, set it to default
    [[ -z "${LOGFILE-}" ]] && LOGFILE="/var/log/idstower/rules_updates.log"

    return 0
}

apiPost() {
    local endpoint=$1
    local instance_name=$2

    logDebug "sending HTTP Post request to API"

    # build the json request with instance name included
    json_request="{\"apikey\": \"${APIKEY}\", \"instanceName\": \"${instance_name}\"}"
    logDebug "API Endpoint: ${APIURL}${endpoint}"
    logDebug "Json Request: ${json_request}"

    # set curl debug option is debug mode is enabled
    curlDebugParameter=""
    if [[ "${debug_flag}" -eq 1 ]]; then
        curlDebugParameter="-v"
    fi

    # send the request
    local apiOutput=""
    apiOutput=$(curl ${curlDebugParameter} -L -s -k --connect-timeout 10 -X POST -H "Content-Type: application/json" --data "${json_request}" ${APIURL}${endpoint} || true) #use `|| true` to ignore curl failures

    logDebug "API Output: ${apiOutput}"

    # make sure that we are getting non-empty response from the api
    if [[ -z "$apiOutput" ]]; then
        die "got empty response from IDSTower API, make sure IDSTower service is running correctly, to debug this issue, re-run this script with --debug option, exiting..."
    fi

    echo "${apiOutput}"
}

# returns suricata binary path
getSuricataBinaryPath() {
    # check if suricata binary exists in the default path
    if [[ -f "/usr/bin/suricata" ]]; then
        suricata_binary_path="/usr/bin/suricata"
    elif [[ -f "/sbin/suricata" ]]; then
        suricata_binary_path="/sbin/suricata"
    elif [[ -f "/usr/sbin/suricata" ]]; then
        suricata_binary_path="/usr/sbin/suricata"
    elif [[ -f "/usr/local/bin/suricata" ]]; then
        suricata_binary_path="/usr/local/bin/suricata"
    elif [[ -f "/usr/local/sbin/suricata" ]]; then
        suricata_binary_path="/usr/local/sbin/suricata"
    else
        logWarning "suricata binary not found, please install suricata or make sure it is in the default path."
        suricata_binary_path=""
    fi

    echo "${suricata_binary_path}"
}

# returns suricatasc binary path
getSuricataScBinaryPath() {
    # Common locations across distributions
    local possible_paths=(
        "/usr/bin/suricatasc"
        "/usr/local/bin/suricatasc"
        "/usr/sbin/suricatasc"
        "/usr/local/sbin/suricatasc"
    )

    # First try 'which' command
    local which_result
    which_result=$(which suricatasc 2>/dev/null)
    if [[ -x "${which_result}" ]]; then
        echo "${which_result}"
        return 0
    fi

    # If 'which' fails, check common locations
    for path in "${possible_paths[@]}"; do
        if [[ -x "${path}" ]]; then
            echo "${path}"
            return 0
        fi
    done

    # If nothing found, return empty string
    echo ""
    return 0
}

# Helper function to get instance-specific config file path
getInstanceConfigFile() {
    local instance_name=$1
    echo "${suricata_config_dir}instance-${instance_name}/suricata.yaml"
}

# Helper function to get the appropriate config file for an instance
# Returns instance-specific config if it exists, otherwise returns default config
# Exits with error if neither can be found
getSuricataConfigFile() {
    local instance_name=$1
    local instance_config_file
    instance_config_file=$(getInstanceConfigFile "${instance_name}")
    local default_config_file="/etc/suricata/suricata.yaml"

    # First try instance-specific config
    if [[ -f "${instance_config_file}" ]]; then
        logDebug "Using instance-specific config for ${instance_name}: ${instance_config_file}"
        echo "${instance_config_file}"
        return 0
    fi

    # Fall back to default config
    if [[ -f "${default_config_file}" ]]; then
        logDebug "Instance-specific config not found for ${instance_name}, using default: ${default_config_file}"
        echo "${default_config_file}"
        return 0
    fi

    # Neither found - this is an error
    die "Neither instance-specific config file (${instance_config_file}) nor default config file (${default_config_file}) could be found for instance ${instance_name}, exiting..."
}

# returns suricata Socket file path
getSuricataSocketFile() {
    # get the passed suricata config file path
    local suricata_config_file="${1-}"
    local instance_name="${2-}"

    # Define the instance-specific socket name pattern
    local instance_socket_name=""
    if [[ -n "${instance_name}" ]]; then
        instance_socket_name="suricata-instance-${instance_name}.socket"
        logDebug "Using instance-specific socket name: ${instance_socket_name}"
    fi

    suricata_binary_path=$(getSuricataBinaryPath)
    logDebug "suricata command binary path: ${suricata_binary_path}"

    # first we check if there is a custom socket file path in the config file
    # check if the return suricata binary exists
    logDebug "checking if suricata binary exists..."
    if [[ -f "${suricata_binary_path}" ]]; then
        logDebug "obtaining socket file path from Suricata config at: ${suricata_config_file}..."
        socket_file=$(${suricata_binary_path} -c "${suricata_config_file}" --dump-config | grep unix-command.filename)
        # if exit code is 0, then there is a custom socket file path in the config file
        if [[ $? -eq 0 ]]; then
            # get the socket file path from the config file
            socket_file=$(echo "${socket_file}" | cut -d "=" -f2 | tr -d ' ')

            # check if socket_file is empty or not
            if [[ -z "${socket_file}" ]]; then
                logDebug "socket file not found in Suricata config file: ${suricata_config_file}"
            else
                logDebug "socket file found in Suricata config file: ${socket_file}"
            fi

            logDebug "checking if socket file exists..."
            # check if socket file exists
            if [[ -S "${socket_file}" ]]; then
                logDebug "Suricata socket file found: ${socket_file}"
            elif [[ -S "/var/${socket_file}" ]]; then
                socket_file="/var/${socket_file}"
                logDebug "socket file found: ${socket_file}"
            elif [[ -S "/var/run/${socket_file}" ]]; then
                socket_file="/var/run/${socket_file}"
                logDebug "socket file found: ${socket_file}"
            elif [[ -S "/var/run/suricata/${socket_file}" ]]; then
                socket_file="/var/run/suricata/${socket_file}"
                logDebug "socket file found: ${socket_file}"
            # Check instance-specific socket paths if an instance name is provided
            elif [[ -n "${instance_name}" ]] && [[ -S "/var/${instance_socket_name}" ]]; then
                socket_file="/var/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/${instance_socket_name}" ]]; then
                socket_file="/var/run/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/suricata/${instance_socket_name}" ]]; then
                socket_file="/var/run/suricata/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            else
                logWarning "socket file not found: ${socket_file}"
                echo ""
                return 0
            fi
        else
            logDebug "socket file not found in Suricata config file: ${suricata_config_file}"
            logDebug "checking if instance-specific or default socket file exists..."

            # Check for instance-specific socket files first if an instance name is provided
            if [[ -n "${instance_name}" ]] && [[ -S "/var/${instance_socket_name}" ]]; then
                socket_file="/var/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/${instance_socket_name}" ]]; then
                socket_file="/var/run/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/suricata/${instance_socket_name}" ]]; then
                socket_file="/var/run/suricata/${instance_socket_name}"
                logDebug "instance socket file found: ${socket_file}"
            else
                logWarning "socket file not found: ${socket_file}"
                echo ""
                return 0
            fi
        fi
    elif [[ -n "${instance_name}" ]] && [[ -S "/var/${instance_socket_name}" ]]; then
        socket_file="/var/${instance_socket_name}"
        logDebug "Suricata binary path not found, using instance socket file path: ${socket_file}"
    elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/${instance_socket_name}" ]]; then
        socket_file="/var/run/${instance_socket_name}"
        logDebug "Suricata binary path not found, using instance socket file path: ${socket_file}"
    elif [[ -n "${instance_name}" ]] && [[ -S "/var/run/suricata/${instance_socket_name}" ]]; then
        socket_file="/var/run/suricata/${instance_socket_name}"
        logDebug "Suricata binary path not found, using instance socket file path: ${socket_file}"
    else
        logWarning "Suricata binary path not found, please install suricata or make sure it is in the default path."
        echo ""
        return 0
    fi

    # after we have got the socket file path, we need to check if it is active (eg: the service is running)
    # to do this we run a simple command and observe the exit code
    # if the exit code is 0, then the socket file is active
    logDebug "getting suricatasc binary path..."
    suricatasc_binary_path=$(getSuricataScBinaryPath)
    if [[ -z "${suricatasc_binary_path}" ]]; then
        logError "Could not find suricatasc executable"
        echo ""
        return 1
    fi

    logDebug "suricatasc command binary path: ${suricatasc_binary_path}"
    logDebug "checking if Suricata socket file at the following location is active: ${socket_file}"

    "${suricatasc_binary_path}" -c version "${socket_file}" >/dev/null 2>&1
    if [[ $? -eq 0 ]]; then
        logDebug "the following socket file is active: ${socket_file}"
        echo "${socket_file}"
        return 0
    else
        logWarning "socket file is not active: ${socket_file}, make sure suricata service is running."
        echo ""
        return 0
    fi
}

# Function to get the rules directory for a specific instance
getInstanceRulesDir() {
    local instance_name=$1
    echo "${suricata_config_dir}instance-${instance_name}/rules/"
}

# Updated function to update Suricata rules for a specific instance
updateSuricataRulesList() {
    local instance_name=$1
    local instance_rules_dir
    instance_rules_dir=$(getInstanceRulesDir "${instance_name}")

    # Call API with instance name
    apiOutput=$(apiPost "hosts/suricata/getIDSRules" "${instance_name}")

    # verify the apiOutput by making sure it contains the header "#IDSTower Auto-generated IDS Rules File"
    if [[ $apiOutput != *"#IDSTower Auto-generated IDS Rules File"* ]]; then
        die "got unexpected response from IDSTower API for instance ${instance_name}, make sure IDSTower service is running correctly, to debug this issue, re-run this script with --debug option, exiting..."
    fi

    # check if apiOutput contains "#Rule Compilation is still under process"
    # if so we exit, since we don't want to reload an empty rule file
    if [[ $apiOutput == *"#Rule Compilation is still under process"* ]]; then
        logWarning "Rule Compilation is still under process in IDSTower host for instance ${instance_name}, will check for rules again in next run..."
        return 1 # Return 1 to indicate no update was done
    fi

    # Define the rules file path for this instance
    local rulesFile="${instance_rules_dir}idstower_suricata.rules"
    logDebug "Local Suricata Rules file for instance ${instance_name}: ${rulesFile}"

    # Check if the instance directory exists, create it if not
    if [[ ! -d "${instance_rules_dir}" ]]; then
        logInfo "Creating rules directory for instance ${instance_name}: ${instance_rules_dir}"
        mkdir -p "${instance_rules_dir}"
        if [[ $? -ne 0 ]]; then
            die "Failed to create rules directory for instance ${instance_name}: ${instance_rules_dir}, exiting..."
        fi
    fi

    # write the api output to a temporary file
    logDebug "writing api output to temporary file for instance ${instance_name}: ${rulesFile}.temporary"
    printf "%s" "$apiOutput" >"${rulesFile}.temporary"

    # calculate the api output file hash values
    logDebug "calculating api output hash for instance ${instance_name}"
    apiOutputHash=$(md5sum "${rulesFile}".temporary | cut -d " " -f1)
    logDebug "Api Output hash for instance ${instance_name}: ${apiOutputHash}"

    # remove the temporary file
    rm -rf "${rulesFile}".temporary

    # make sure the rules file exist
    touch "${rulesFile}"

    # calculate the local rules file hash
    localSuricataRulesFileHash=$(md5sum "${rulesFile}" | cut -d " " -f1)
    logDebug "Local Suricata rules file hash for instance ${instance_name}: ${localSuricataRulesFileHash}"

    if [[ $apiOutputHash == "$localSuricataRulesFileHash" ]]; then
        # exit only if force_flag is not set
        if [[ $force_flag -ne 1 ]]; then
            logInfo "Local Suricata Rules file for instance ${instance_name} is up-to-date."
            return 1 # Return 1 to indicate no update was needed
        fi
    fi

    logInfo "Updating Local Suricata Rules file for instance ${instance_name}: ${rulesFile}"
    printf "%s" "$apiOutput" >"${rulesFile}"

    if [[ $? -ne 0 ]]; then
        die "Failed to update Local Suricata Rules file for instance ${instance_name}, could not write to: ${rulesFile}, exiting..."
    fi

    logInfo "Local Suricata Rules file for instance ${instance_name} updated successfully."
    return 0 # Return 0 to indicate rules were updated
}

# Function to get the service name for a specific instance
getSuricataServiceName() {
    local instance_name=$1
    echo "suricata-instance-${instance_name}.service"
}

restartSuricataService() {
    local instance_name=$1
    local service_name
    service_name=$(getSuricataServiceName "${instance_name}")

    # we will only attempt to restart suricata if the service is already running
    # we do this to prevent the script from starting suricata service while it was in stopped state
    serviceStatus=$(systemctl show -p ActiveState "${service_name}")

    if [[ "$serviceStatus" != "ActiveState=active" ]]; then
        logInfo "Suricata service ${service_name} is not running, skipping restart."
        return 0
    fi

    systemctl restart "${service_name}"

    if [[ $? -ne 0 ]]; then
        logError "Failed to restart Suricata service ${service_name}, to investigate why restart failed, run: sudo systemctl status ${service_name} -l , Or run: sudo journalctl -ex -u ${service_name}"
        return 1
    fi

    logInfo "Suricata service ${service_name} restarted successfully."
    return 0
}

reloadSuricataRules() {
    local instance_name=$1
    local suricata_config_file=$2
    local service_name
    service_name=$(getSuricataServiceName "${instance_name}")

    # we will only attempt to reload suricata if the service is already running
    # we do this to prevent the script from starting suricata service while it was in stopped state
    serviceStatus=$(systemctl show -p ActiveState "${service_name}")

    if [[ "$serviceStatus" != "ActiveState=active" ]]; then
        logInfo "Suricata service ${service_name} is not running, skipping reload."
        return 0
    fi

    # get socket file path
    socket_file=$(getSuricataSocketFile "${suricata_config_file}" "${instance_name}")
    if [[ -z "${socket_file}" ]]; then
        logWarning "Failed to get Suricata socket file path for instance ${instance_name}, skipping reload."
        return 0
    fi

    # get suricatasc binary path
    logDebug "getting suricatasc binary path..."
    suricatasc_binary_path=$(getSuricataScBinaryPath)
    logDebug "suricatasc command binary path: ${suricatasc_binary_path}"
    if [[ -z "${suricatasc_binary_path}" ]]; then
        logError "Failed to get Suricatasc binary path for instance ${instance_name}, skipping reload."
        return 0
    fi

    "${suricatasc_binary_path}" -c "ruleset-reload-rules" "${socket_file}" >/dev/null 2>&1

    if [[ $? -ne 0 ]]; then
        logError "Failed to reload Suricata rules for instance ${instance_name}, to investigate why reload failed, run: sudo systemctl status ${service_name} -l , Or run: sudo journalctl -ex -u ${service_name} and view Suricata logs."
        return 1
    fi

    logInfo "Suricata rules for instance ${instance_name} reloaded successfully."
    return 0
}

# Function to process a single instance
processInstance() {
    local instance_name=$1
    local updated=0 # Flag to track if any updates were made

    logInfo "===== Processing instance: ${instance_name} ====="

    # Get the appropriate config file for this instance (instance-specific or default)
    local suricata_config_file
    suricata_config_file=$(getSuricataConfigFile "${instance_name}")

    # Update rules for this instance
    updateSuricataRulesList "${instance_name}" || true
    if [[ $? -eq 0 ]]; then
        updated=1
    fi

    # If any updates were made, reload the rules
    if [[ $updated -eq 1 ]] || [[ $force_flag -eq 1 ]]; then
        logInfo "Reloading Suricata rules for instance ${instance_name}..."
        reloadSuricataRules "${instance_name}" "${suricata_config_file}" || true
    else
        logInfo "No updates needed for instance ${instance_name}."
    fi

    logInfo "===== Finished processing instance: ${instance_name} ====="
}

############## END Utility functions ##############

############## Start Main Script ##############
parse_params "$@"
setup_colors

logInfo "==================== Suricata Rules update script ===================="
logInfo "Starting script..."

# Parse instances from the comma-separated list
IFS=',' read -ra INSTANCE_ARRAY <<<"$INSTANCES"
logInfo "Found ${#INSTANCE_ARRAY[@]} instances to process: ${INSTANCES}"

# skip the random delay if skip_delay_flag is set
if [[ $skip_delay_flag -eq 0 ]]; then
    logInfo "Sleeping for up to 120 seconds to prevent overwhelming IDSTower, to skip this, re-run script with --skip-delay"
    sleep "$(shuf -i 0-120 -n 1)"
fi

# Process each instance
for instance_name in "${INSTANCE_ARRAY[@]}"; do
    # Trim whitespace from instance name
    instance_name=$(echo "$instance_name" | xargs)

    if [[ -z "$instance_name" ]]; then
        logWarning "Empty instance name found in list, skipping"
        continue
    fi

    # Process this instance
    processInstance "$instance_name"
done

logInfo "All instances processed successfully."
logInfo "==================== Suricata Rules update script completed ===================="
