#!/usr/bin/env bash
# shellcheck disable=SC2006
# fail the whole script if a command fails
set -Eeuo pipefail

cleanup() {
  trap '' EXIT INT TERM ERR
  # script cleanup here
}

trap 'cleanup' EXIT INT TERM ERR

############## Start Utility functions ##############
logToFile() {
  MAX_LOG_LINES=200000
  LINES_TO_REMOVE=10000
  # create the log file directory if it does not exist
  if [[ ! -d "${LOGFILE%/*}" ]]; then
    mkdir -p "${LOGFILE%/*}"
  fi

  # Check if the log file exists
  if [[ -f "$LOGFILE" ]]; then
    # Count the number of lines in the log file
    num_lines=$(wc -l <"$LOGFILE")

    # Check if the number of lines exceeds the limit
    if ((num_lines >= MAX_LOG_LINES)); then
      # Calculate the number of lines to remove
      lines_to_remove=$((num_lines - MAX_LOG_LINES + LINES_TO_REMOVE))
      # Remove the older lines from the log file
      sed -i "1,${lines_to_remove}d" "$LOGFILE"

      logInfo "Truncated log file $LOGFILE."
    fi
  fi

  echo -e "$1" >>"${LOGFILE}"
}

logInfo() {
  local calling_function_name=${FUNCNAME[1]}
  if [[ "${debug_flag}" -eq 1 ]]; then
    calling_function_string="${calling_function_name}(): "
  else
    calling_function_string=""
  fi

  # check if /dev/tty is available by making sure STDOUT is attached to TTY
  if [ -t 1 ] || [ -t 2 ]; then
    echo -e "[$(date +"%Y-%m-%dT%H:%M:%S%z")] ${BLUE}[Info] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" >/dev/tty
  fi

  logToFile "[$(date +"%Y-%m-%dT%H:%M:%S%z")] [Info] ${calling_function_string} $1"
}

logWarning() {
  local calling_function_name=${FUNCNAME[1]}
  if [[ "${debug_flag}" -eq 1 ]]; then
    calling_function_string="${calling_function_name}(): "
  else
    calling_function_string=""
  fi

  # check if /dev/tty is available by making sure STDOUT is attached to TTY
  if [ -t 1 ] || [ -t 2 ]; then
    echo -e "[$(date +"%Y-%m-%dT%H:%M:%S%z")] ${ORANGE}[Warning] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" >/dev/tty
  fi

  logToFile "[$(date +"%Y-%m-%dT%H:%M:%S%z")] [Warning] ${calling_function_string} $1"
}

logError() {
  local calling_function_name=${FUNCNAME[1]}
  if [[ "${debug_flag}" -eq 1 ]]; then
    calling_function_string="${calling_function_name}(): "
  else
    calling_function_string=""
  fi

  # check if /dev/tty is available by making sure STDOUT is attached to TTY
  if [ -t 1 ] || [ -t 2 ]; then
    echo -e "[$(date +"%Y-%m-%dT%H:%M:%S%z")] ${RED}[Error] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" >/dev/tty
  fi

  logToFile "[$(date +"%Y-%m-%dT%H:%M:%S%z")] [Error] ${calling_function_string} $1"
}

logDebug() {
  local calling_function_name=${FUNCNAME[1]}
  if [[ "${debug_flag}" -eq 1 ]]; then
    calling_function_string="${calling_function_name}(): "
  else
    calling_function_string=""
  fi

  if [[ "${debug_flag}" -eq 1 ]]; then
    # check if /dev/tty is available by making sure STDOUT is attached to TTY
    if [ -t 1 ] || [ -t 2 ]; then
      echo -e "[$(date +"%Y-%m-%dT%H:%M:%S%z")] ${PURPLE}[Debug] ${NOFORMAT}${GREEN}${calling_function_string}$1${NOFORMAT}" >/dev/tty
    fi
    logToFile "[$(date +"%Y-%m-%dT%H:%M:%S%z")] [Debug] ${calling_function_string}$1"
  fi
}

die() {
  local msg=$1
  local code=${2-1} # default exit status 1
  logError "$msg"
  exit "$code"
}

setup_colors() {
  if [[ -t 2 ]] && [[ -z "${NO_COLOR-}" ]] && [[ "${TERM-}" != "dumb" ]]; then
    NOFORMAT='\033[0m' RED='\033[0;31m' GREEN='\033[0;32m' ORANGE='\033[0;33m' BLUE='\033[0;34m' PURPLE='\033[0;35m' CYAN='\033[0;36m' YELLOW='\033[1;33m'
  else
    NOFORMAT='' RED='' GREEN='' ORANGE='' BLUE='' PURPLE='' CYAN='' YELLOW=''
  fi
}

usage() {
  cat <<EOF
IDSTower Suricata logs deletion script, this script deletes old & unused Suricata logs to prevent the disk from getting full.
The script is normally run by Cron periodically, but can be run manually run for debugging purposes.

Usage: $(basename "${BASH_SOURCE[0]}") [-u] [-k] [-d] [-h] [--no-color]
Available options:

-p, --path  Suricata logs directory path (default: /var/log/suricata)
-u, --disk-utilization-threshold   Disk utilization threshold in percentage (default: 50), this option
                                   takes precedence over --keep-days. When disk utilization is above
                                   this threshold, the script will delete Suricata logs (oldest first)
                                   regardless of age until disk utilization is below this threshold
-k, --keep-days   Number of days to keep logs (default: 90), the script will delete Suricata logs older than
                  this number of days, regardless of if disk utilization is above the threshold or not
-x, --dry-run     Do not delete logs, just print the list of logs that will be deleted
-l, --log-file  Log file path (default: /var/log/idstower/logs_deletion.log)
-d, --debug     Print script debug info
-h, --help  Print this help and exit
--no-color      Do not use colors in output

EOF
  exit
}

parse_params() {
  # default values of variables
  dry_run_flag=0
  debug_flag=0

  while :; do
    case "${1-}" in
    -h | --help) usage ;;
    -x | --dry-run) dry_run_flag=1 ;;
    -d | --debug) debug_flag=1 ;;
    --no-color) NO_COLOR=1 ;;
    -p | --path)
      SURICATA_LOGS_DIR_PATH="${2-}"
      shift
      ;;
    -u | --disk-utilization-threshold)
      DISK_UTILIZATION_THRESHOLD="${2-}"
      # check if DISK_UTILIZATION_THRESHOLD is a number
      if ! [[ "${DISK_UTILIZATION_THRESHOLD}" =~ ^[0-9]+$ ]]; then
        echo -e "disk utilization threshold must be a number" && exit 1
      fi
      # check if DISK_UTILIZATION_THRESHOLD is between 0 and 100
      if [[ "${DISK_UTILIZATION_THRESHOLD}" -lt 0 ]] || [[ "${DISK_UTILIZATION_THRESHOLD}" -gt 100 ]]; then
        echo -e "disk utilization threshold must be between 0 and 100" && exit 1
      fi
      shift
      ;;
    -k | --keep-days)
      KEEP_DAYS="${2-}"
      # check if KEEP_DAYS is a positive number
      if ! [[ "${KEEP_DAYS}" =~ ^[0-9]+$ ]]; then
        echo -e "keep days must be a positive number" && exit 1
      fi
      shift
      ;;
    -l | --log-file)
      LOGFILE="${2-}"
      shift
      ;;
    -?*)
      echo "ERROR: Unknown option: $1" &
      usage
      ;;
    *) break ;;
    esac
    shift
  done

  # set default values
  [[ -z "${SURICATA_LOGS_DIR_PATH-}" ]] && SURICATA_LOGS_DIR_PATH="/var/log/suricata"
  [[ -z "${DISK_UTILIZATION_THRESHOLD-}" ]] && DISK_UTILIZATION_THRESHOLD="50"
  [[ -z "${KEEP_DAYS-}" ]] && KEEP_DAYS="90"
  [[ -z "${LOGFILE-}" ]] && LOGFILE="/var/log/idstower/logs_deletion.log"

  return 0
}

getDiskUtilization() {
  # get passed disk path
  local disk_path="${1}"
  logDebug "Getting disk utilization for ${disk_path}"
  # get disk utilization
  local disk_utilization=""
  disk_utilization=$(df -h "${disk_path}" | tail -1 | awk '{print $5}' | sed 's/%//g')
  logDebug "Disk utilization for ${disk_path} is: ${disk_utilization}%"
  echo "${disk_utilization}"
}

deleteSuricataLogs() {
  # set passed values
  local suricata_logs_dir_path="${1}"
  local disk_utilization_threshold="${2}"
  local keep_days="${3}"
  local dry_run_flag="${4}"

  # check that Suricata logs directory exists
  logDebug "Verifying that Suricata logs directory exists: ${suricata_logs_dir_path}"
  if [[ ! -d "${suricata_logs_dir_path}" ]]; then
    die "Provided Suricata logs directory does not exist: ${suricata_logs_dir_path}, existing..."
  fi

  local disk_utilization
  disk_utilization=$(getDiskUtilization "${suricata_logs_dir_path}")
  logInfo "Current Disk utilization for ${suricata_logs_dir_path} is: ${disk_utilization}%"

  # loop through the list of Suricata logs and remove the ones that are not used by any process
  find "${suricata_logs_dir_path}" -type f -printf '%T@ %p\0' | sort -zn | while IFS= read -r -d '' entry; do
    # check if disk utilization is below threshold
    disk_utilization=$(getDiskUtilization "${suricata_logs_dir_path}")
    if [[ "${disk_utilization}" -lt "${disk_utilization_threshold}" ]]; then
      logInfo "Current Disk utilization for ${suricata_logs_dir_path} is: ${disk_utilization}%"
      logInfo "Disk utilization is below threshold."
      break
    else
      logDebug "Current Disk utilization for ${suricata_logs_dir_path} is: ${disk_utilization}%"
      logDebug "Disk utilization is above threshold, continuing deletion of Suricata logs..."
    fi

    file="${entry#* }"
    logDebug "Checking if Suricata log is in use: ${file}"
    local suricata_log_in_use
    # Note: lsof returns exit code 1 when no processes have the file open, which is actually
    # the desired state (file not in use = safe to delete). However, with 'set -Eeuo pipefail'
    # enabled at the script level, any non-zero exit code in a pipeline would cause the script
    # to terminate. The '|| true' appended here ensures the command substitution always succeeds
    # regardless of lsof's exit code. When the file is NOT in use (exit code 1), lsof produces
    # no output, resulting in an empty suricata_log_in_use variable, which is correct.
    suricata_log_in_use=$(lsof -F n "${file}" 2>/dev/null | grep -v '^p' | sed 's/^n//g' || true)
    if [[ -z "${suricata_log_in_use}" ]]; then
      # check if dry run flag is set
      if [[ "${dry_run_flag}" -eq 1 ]]; then
        logInfo "Dry run flag is set, file ${file} will not be deleted."
      else
        logInfo "Deleting Suricata log: ${file} ..."
        rm -f "${file}"
      fi
    else
      # If disk usage is >= 90%, delete the file even if it's in use
      if [[ "${disk_utilization}" -ge 90 ]]; then
        if [[ "${dry_run_flag}" -eq 1 ]]; then
          logWarning "Dry run flag is set, but disk usage is critical (${disk_utilization}%). File ${file} would be forcefully deleted despite being in use."
          logWarning "IMPORTANT: Configure your logshipper to release file handles after shipping logs to prevent disk from getting full and avoid forced deletions."
        else
          logWarning "Disk usage is critical (${disk_utilization}%). Forcefully deleting Suricata log: ${file} even though it is still in use to prevent disk from getting full."
          logWarning "IMPORTANT: Configure your logshipper to release file handles after shipping logs to prevent this situation in the future."
          rm -f "${file}"
        fi
      else
        logWarning "Suricata log: ${file} is still in use, will not delete it. Make sure your logshipper is configured to release file handles after shipping logs to prevent disk from getting full."
      fi
    fi
  done || true # avoid pipefail abort when break closes the loop early

  # Delete Suricata logs older than KEEP_DAYS days
  logInfo "Deleting Suricata logs older than ${keep_days} days..."
  find "${suricata_logs_dir_path}" -type f -mtime +"${keep_days}" -printf '%T@ %p\0' | sort -zn | while IFS= read -r -d '' entry; do
    file="${entry#* }"
    logDebug "Checking if Suricata log is in use: ${file}"
    local suricata_log_in_use
    disk_utilization=$(getDiskUtilization "${suricata_logs_dir_path}")
    # Note: lsof returns exit code 1 when no processes have the file open, which is actually
    # the desired state (file not in use = safe to delete). However, with 'set -Eeuo pipefail'
    # enabled at the script level, any non-zero exit code in a pipeline would cause the script
    # to terminate. The '|| true' appended here ensures the command substitution always succeeds
    # regardless of lsof's exit code. When the file is NOT in use (exit code 1), lsof produces
    # no output, resulting in an empty suricata_log_in_use variable, which is correct.
    suricata_log_in_use=$(lsof -F n "${file}" 2>/dev/null | grep -v '^p' | sed 's/^n//g' || true)
    if [[ -z "${suricata_log_in_use}" ]]; then
      # check if dry run flag is set
      if [[ "${dry_run_flag}" -eq 1 ]]; then
        logInfo "Dry run flag is set, file ${file} will not be deleted."
      else
        logInfo "Deleting Suricata log: ${file} ..."
        rm -f "${file}"
      fi
    else
      # If disk usage is >= 90%, delete the file even if it's in use
      if [[ "${disk_utilization}" -ge 90 ]]; then
        if [[ "${dry_run_flag}" -eq 1 ]]; then
          logWarning "Dry run flag is set, but disk usage is critical (${disk_utilization}%). File ${file} would be forcefully deleted despite being in use."
          logWarning "IMPORTANT: Configure your logshipper to release file handles after shipping logs to prevent disk from getting full and avoid forced deletions."
        else
          logWarning "Disk usage is critical (${disk_utilization}%). Forcefully deleting Suricata log: ${file} even though it is still in use to prevent disk from getting full."
          logWarning "IMPORTANT: Configure your logshipper to release file handles after shipping logs to prevent this situation in the future."
          rm -f "${file}"
        fi
      else
        logWarning "Suricata log: ${file} is still in use, will not delete it. Make sure your logshipper is configured to release file handles after shipping logs to prevent disk from getting full."
      fi
    fi
  done || true # avoid pipefail abort when break closes the loop early
}

############## END Utility functions ##############

############## Start Main Script ##############
parse_params "$@"
setup_colors

logInfo "==================== Suricata logs deletion script ===================="
logInfo "Starting script..."

logDebug "Passed parameters: "
logDebug "SURICATA_LOGS_DIR_PATH: ${SURICATA_LOGS_DIR_PATH}"
logDebug "DISK_UTILIZATION_THRESHOLD: ${DISK_UTILIZATION_THRESHOLD}"
logDebug "KEEP_DAYS: ${KEEP_DAYS}"
logDebug "dry_run_flag: ${dry_run_flag}"

# check if lsof command is installed
if ! command -v lsof &>/dev/null; then
  die "lsof command is not installed, please install it before running this script, exiting..."
fi

# delete old Suricata logs
deleteSuricataLogs "${SURICATA_LOGS_DIR_PATH}" "${DISK_UTILIZATION_THRESHOLD}" "${KEEP_DAYS}" "${dry_run_flag}"

logInfo "Finished script"
